//  PCANTP_2004.pas
//
//  ~~~~~~~~~~~~
//
//  PCAN-ISO-TP API
//
//  ~~~~~~~~~~~~
//
//  ------------------------------------------------------------------
//  Last changed by:    $Author: Fabrice $
//  Last changed date:  $Date: 2025-01-23 11:32:04 +0100 (Thu, 23 Jan 2025) $
//
//  Language: Pascal OO
//  ------------------------------------------------------------------
//
//  Copyright (C) 2015  PEAK-System Technik GmbH, Darmstadt
//  more Info at http://www.peak-system.com
//
unit PCANTP_2004;

interface

type
    TPCANTPHandle = Word;
    TPCANTPBitrateFD = PAnsiChar;

{$Z2}
    /// <summary>
    /// Represents a PCAN Baud rate register value
    /// </summary>
    TPCANTPBaudrate =  (
        /// <summary>
        /// 1 MBit/s
        /// </summary>
        PCANTP_BAUD_1M = $0014,
        /// <summary>
        /// 800 kBit/s
        /// </summary>
        PCANTP_BAUD_800K = $0016,
        /// <summary>
        /// 500 kBit/s
        /// </summary>
        PCANTP_BAUD_500K = $001C,
        /// <summary>
        /// 250 kBit/s
        /// </summary>
        PCANTP_BAUD_250K = $011C,
        /// <summary>
        /// 125 kBit/s
        /// </summary>
        PCANTP_BAUD_125K = $031C,
        /// <summary>
        /// 100 kBit/s
        /// </summary>
        PCANTP_BAUD_100K = $432F,
        /// <summary>
        /// 95,238 kBit/s
        /// </summary>
        PCANTP_BAUD_95K = $C34E,
        /// <summary>
        /// 83,333 kBit/s
        /// </summary>
        PCANTP_BAUD_83K = $852B,
        /// <summary>
        /// 50 kBit/s
        /// </summary>
        PCANTP_BAUD_50K = $472F,
        /// <summary>
        /// 47,619 kBit/s
        /// </summary>
        PCANTP_BAUD_47K = $1414,
        /// <summary>
        /// 33,333 kBit/s
        /// </summary>
        PCANTP_BAUD_33K = $8B2F,
        /// <summary>
        /// 20 kBit/s
        /// </summary>
        PCANTP_BAUD_20K = $532F,
        /// <summary>
        /// 10 kBit/s
        /// </summary>
        PCANTP_BAUD_10K = $672F,
        /// <summary>
        /// 5 kBit/s
        /// </summary>
        PCANTP_BAUD_5K = $7F7F
    );

{$Z1}
    /// <summary>
    /// Represents the different Not Plug-And-Play PCAN Hardware types
    /// </summary>
    TPCANTPHWType = (
        /// <summary>
        /// PCAN-ISA 82C200
        /// </summary>
        PCANTP_TYPE_ISA = $01,
        /// <summary>
        /// PCAN-ISA SJA1000
        /// </summary>
        PCANTP_TYPE_ISA_SJA = $09,
        /// <summary>
        /// PHYTEC ISA
        /// </summary>
        PCANTP_TYPE_ISA_PHYTEC = $04,
        /// <summary>
        /// PCAN-Dongle 82C200
        /// </summary>
        PCANTP_TYPE_DNG = $02,
        /// <summary>
        /// PCAN-Dongle EPP 82C200
        /// </summary>
        PCANTP_TYPE_DNG_EPP = $03,
        /// <summary>
        /// PCAN-Dongle SJA1000
        /// </summary>
        PCANTP_TYPE_DNG_SJA = $05,
        /// <summary>
        /// PCAN-Dongle EPP SJA1000
        /// </summary>
        PCANTP_TYPE_DNG_SJA_EPP = $06
    );

{$Z4}
    /// <summary>
    /// Represent the PCANTP error and status codes
    /// </summary>
    TPCANTPStatus = (
        /// <summary>
        /// No error
        /// </summary>
        PCANTP_ERROR_OK = $00000,
        /// <summary>
        /// Not Initialized
        /// </summary>
        PCANTP_ERROR_NOT_INITIALIZED = $00001,
        /// <summary>
        /// Already Initialized
        /// </summary>
        PCANTP_ERROR_ALREADY_INITIALIZED = $00002,
        /// <summary>
        /// Could not obtain memory
        /// </summary>
        PCANTP_ERROR_NO_MEMORY = $00003,
        /// <summary>
        /// Input buffer overflow
        /// </summary>
        PCANTP_ERROR_OVERFLOW = $00004,
        /// <summary>
        /// No Message available
        /// </summary>
        PCANTP_ERROR_NO_MESSAGE = $00007,
        /// <summary>
        /// Wrong message parameters
        /// </summary>
        PCANTP_ERROR_WRONG_PARAM = $00008,
        /// <summary>
        /// PCANTP Channel is in BUS-LIGHT error state
        /// </summary>
        PCANTP_ERROR_BUSLIGHT = $00009,
        /// <summary>
        /// PCANTP Channel is in BUS-HEAVY error state
        /// </summary>
        PCANTP_ERROR_BUSHEAVY = $0000A,
        /// <summary>
        /// PCANTP Channel is in BUS-OFF error state
        /// </summary>
        PCANTP_ERROR_BUSOFF = $0000B,
        /// <summary>
        /// Global CAN error, status code for composition of PCANBasic Errors.
        /// Remove this value to get a PCAN-Basic TPCANStatus error code.
        /// </summary>
        PCANTP_ERROR_CAN_ERROR = LongWord($80000000)
    );

{$Z1}
    /// <summary>
    /// Represent message request confirmation values defined in ISO 15765-2
    /// </summary>
    TPCANTPConfirmation = (
        /// <summary>
        /// No network error
        /// </summary>
        PCANTP_N_OK = $00,
        /// <summary>
        /// timeout occured between 2 frames transmission (sender and receiver side)
        /// </summary>
        PCANTP_N_TIMEOUT_A = $01,
        /// <summary>
        /// sender side timeout while waiting for flow control frame
        /// </summary>
        PCANTP_N_TIMEOUT_BS = $02,
        /// <summary>
        /// receiver side timeout while waiting for consecutive frame
        /// </summary>
        PCANTP_N_TIMEOUT_CR = $03,
        /// <summary>
        /// unexpected sequence number
        /// </summary>
        PCANTP_N_WRONG_SN = $04,
        /// <summary>
        /// invalid or unknown FlowStatus
        /// </summary>
        PCANTP_N_INVALID_FS = $05,
        /// <summary>
        /// unexpected protocol data unit
        /// </summary>
        PCANTP_N_UNEXP_PDU = $06,
        /// <summary>
        /// reception of flow control WAIT frame that exceeds the maximum counter defined by PCANTP_PARAM_WFT_MAX
        /// </summary>
        PCANTP_N_WFT_OVRN = $07,
        /// <summary>
        /// buffer on the receiver side cannot store the data length (server side only)
        /// </summary>
        PCANTP_N_BUFFER_OVFLW = $08,
        /// <summary>
        /// general error
        /// </summary>
        PCANTP_N_ERROR = $09
    );

    /// <summary>
    /// PCANTP parameters
    /// </summary>
    TPCANTPParameter = (
        /// <summary>
        /// 1 BYTE data describing the block size parameter (BS)
        /// </summary>
        PCANTP_PARAM_BLOCK_SIZE = $E1,
        /// <summary>
        /// 1 BYTE data describing the seperation time parameter (STmin)
        /// </summary>
        PCANTP_PARAM_SEPARATION_TIME = $E2,
        /// <summary>
        /// 1 BYTE data describing the debug mode
        /// </summary>
        PCANTP_PARAM_DEBUG = $E3,
        /// <summary>
        /// 1 Byte data describing the condition of a channel
        /// </summary>
        PCANTP_PARAM_CHANNEL_CONDITION = $E4,
        /// <summary>
        /// Integer data describing the Wait Frame Transmissions parameter.
        /// </summary>
        PCANTP_PARAM_WFT_MAX = $E5,
        /// <summary>
        /// 1 BYTE data stating if pending messages are displayed/hidden
        /// </summary>
        PCANTP_PARAM_MSG_PENDING = $E6,
        /// <summary>
        /// PCAN-ISO-TP API version parameter
        /// </summary>
        PCANTP_PARAM_API_VERSION = $E7,
        /// <summary>
        /// 1 BYTE data stating if CAN frame DLC uses padding or not
        /// </summary>
        PCANTP_PARAM_CAN_DATA_PADDING = $E8,
        /// <summary>
        ///  1 BYTE data stating if unsegmented (NON-ISO-TP) CAN frames can be received
        /// </summary>
        PCANTP_PARAM_CAN_UNSEGMENTED = $E9,
        /// <summary>
        /// PCAN-ISO-TP receive event handler parameter
        /// </summary>
        PCANTP_PARAM_RECEIVE_EVENT = $EA,
        /// <summary>
        /// 1 BYTE data stating the value used for CAN data padding
        /// </summary>
        PCANTP_PARAM_PADDING_VALUE = $ED,
        /// <summary>
        /// 1 BYTE data stating the default priority value for normal fixed, mixed and enhanced addressing (default=6)
        /// </summary>
        PCANTP_PARAM_J1939_PRIORITY = $EE,
        /// <summary>
        /// 1 BYTE data stating the default DLC to use when transmitting messages with CAN FD
        /// </summary>
        PCANTP_PARAM_CAN_TX_DL = $EF,
        /// <summary>
        /// 1 BYTE data stating how to optimize Minimum SeparationTime latency (enabled=1 by default).
        /// Set value to 0 to ensure STMin is always strictly respected (but consequently lose some speed in communication process).
        /// </summary>
        PCANTP_PARAM_SEPARATION_TIME_OPTIMIZATION = $F0
    );

    /// <summary>
    /// PCANTP message types
    /// </summary>
    TPCANTPMessageType = (
        /// <summary>
        /// Unknown (non-ISO-TP) message
        /// </summary>
        PCANTP_MESSAGE_UNKNOWN = $00,
        /// <summary>
        /// Diagnostic Request/Confirmation
        /// </summary>
        PCANTP_MESSAGE_DIAGNOSTIC = $01,
        /// <summary>
        /// Remote Dignostic Request/Confirmation (uses RA address)
        /// </summary>
        PCANTP_MESSAGE_REMOTE_DIAGNOSTIC = $02,
        /// <summary>
        /// Confirms that a message has been sent successfully/ not successfully
        /// </summary>
        PCANTP_MESSAGE_REQUEST_CONFIRMATION = $03,
        /// <summary>
        /// Multi-Frame Message is being received
        /// </summary>
        PCANTP_MESSAGE_INDICATION = $04,
        /// <summary>
        /// Multi-Frame Message is being transmitted
        /// </summary>
        PCANTP_MESSAGE_INDICATION_TX = $05
    );

    /// <summary>
    /// PCANTP CAN ID types
    /// </summary>
    TPCANTPIdType = (
        /// <summary>
        /// 11 bits CAN ID (CAN Standard Frame)
        /// </summary>
        PCANTP_ID_CAN_11BIT = $01,
        /// <summary>
        /// 29 bits CAN ID (CAN Extended Frame)
        /// </summary>
        PCANTP_ID_CAN_29BIT = $02,
        /// <summary>
        /// CAN FD flag
        /// </summary>
        PCANTP_ID_CAN_FD = $04,
        /// <summary>
        /// Bitrate Switch flag (only if CAN FD)
        /// </summary>
        PCANTP_ID_CAN_BRS = $08
    );

    /// <summary>
    /// PCANTP addressing type
    /// </summary>
    TPCANTPFormatType = (
        /// <summary>
        /// unknown adressing format
        /// </summary>
        PCANTP_FORMAT_UNKNOWN = $FF,
        /// <summary>
        /// unsegmented CAN frame
        /// </summary>
        PCANTP_FORMAT_NONE = $00,
        /// <summary>
        /// normal adressing format from ISO 15765-2
        /// </summary>
        PCANTP_FORMAT_NORMAL = $01,
        /// <summary>
        /// fixed normal adressing format from ISO 15765-2
        /// </summary>
        PCANTP_FORMAT_FIXED_NORMAL = $02,
        /// <summary>
        /// extended adressing format from ISO 15765-2
        /// </summary>
        PCANTP_FORMAT_EXTENDED = $03,
        /// <summary>
        /// mixed adressing format from ISO 15765-2
        /// </summary>
        PCANTP_FORMAT_MIXED = $04,
        /// <summary>
        /// enhanced adressing format from ISO 15765-3
        /// </summary>
        PCANTP_FORMAT_ENHANCED = $05
    );

    /// <summary>
    /// PCANTP addressing type
    /// </summary>
    TPCANTPAddressingType = (
        /// <summary>
        /// Unknown (non-ISO-TP) message
        /// </summary>
        PCANTP_ADDRESSING_UNKNOWN = $00,
        /// <summary>
        /// Physical addressing
        /// </summary>
        PCANTP_ADDRESSING_PHYSICAL = $01,
        /// <summary>
        /// Functional addressing
        /// </summary>
        PCANTP_ADDRESSING_FUNCTIONAL = $02
    );

    /// <summary>
    /// A CAN-TP Message
    /// </summary>
    TPCANTPMsg = record
        /// <summary>
        /// Represents the origin of this message (address from
        /// where this message was or will be sent)
        /// </summary>
        SA: Byte;
        /// <summary>
        /// Represents the destination of this message (address to
        /// where this message was or will be sent)
        /// </summary>
        TA: Byte;
        /// <summary>
        /// Represents the kind of addressing being used for communication
        /// </summary>
        TA_TYPE: TPCANTPAddressingType;
        /// <summary>
        /// Represents the destination of this message in a remote network
        /// </summary>
        RA: Byte;

        /// <summary>
        /// Identifies the kind of CAN Identifier
        /// </summary>
        IDTYPE: TPCANTPIdType;
        /// <summary>
        /// Identifies the kind of data represented by this message
        /// </summary>
        MSGTYPE: TPCANTPMessageType;
        /// <summary>
        /// Represents the kind of format represented by this message
        /// </summary>
        FORMAT: TPCANTPFormatType;

        /// <summary>
        /// Represents the buffer containing the data of this message
        /// </summary>
        DATA: array[0..4094] of Byte;
        /// <summary>
        /// Indicates the length of the useful data within the DATA buffer
        /// </summary>
        LEN: Word;
        /// <summary>
        /// Represent message request confirmation
        /// </summary>
        RESULT: TPCANTPConfirmation;
    end;

    // Represents a timestamp of a received PCANTP message.
    // Total Microseconds = micros + 1000 * millis + 0x100000000 * 1000 * millis_overflow
    //
    TPCANTPTimestamp = record
        millis: Longword;          // Base-value: milliseconds: 0.. 2^32-1
        millis_overflow: Word;     // Roll-arounds of millis
        micros: Word;              // Microseconds: 0..999
    end;
    PTPCANTPTimestamp = ^TPCANTPTimestamp;

    /// <summary>
    /// The PCAN-ISO-TP API class
    /// </summary>
    TCanTpApi = class
    public
        class var
            /// <summary>
            /// Undefined/default value for a PCAN bus
            /// </summary>
            const PCANTP_NONEBUS: TPCANTPHandle = $00;

            /// <summary>
            /// PCAN-ISA interface, channel 1
            /// </summary>
            const PCANTP_ISABUS1: TPCANTPHandle = $21;
            /// <summary>
            /// PCAN-ISA interface, channel 2
            /// </summary>
            const PCANTP_ISABUS2: TPCANTPHandle = $22;
            /// <summary>
            /// PCAN-ISA interface, channel 3
            /// </summary>
            const PCANTP_ISABUS3: TPCANTPHandle = $23;
            /// <summary>
            /// PCAN-ISA interface, channel 4
            /// </summary>
            const PCANTP_ISABUS4: TPCANTPHandle = $24;
            /// <summary>
            /// PCAN-ISA interface, channel 5
            /// </summary>
            const PCANTP_ISABUS5: TPCANTPHandle = $25;
            /// <summary>
            /// PCAN-ISA interface, channel 6
            /// </summary>
            const PCANTP_ISABUS6: TPCANTPHandle = $26;
            /// <summary>
            /// PCAN-ISA interface, channel 7
            /// </summary>
            const PCANTP_ISABUS7: TPCANTPHandle = $27;
            /// <summary>
            /// PCAN-ISA interface, channel 8
            /// </summary>
            const PCANTP_ISABUS8: TPCANTPHandle = $28;

            /// <summary>
            /// PPCAN-Dongle/LPT interface, channel 1
            /// </summary>
            const PCANTP_DNGBUS1: TPCANTPHandle = $31;

            /// <summary>
            /// PCAN-PCI interface, channel 1
            /// </summary>
            const PCANTP_PCIBUS1: TPCANTPHandle = $41;
            /// <summary>
            /// PCAN-PCI interface, channel 2
            /// </summary>
            const PCANTP_PCIBUS2: TPCANTPHandle = $42;
            /// <summary>
            /// PCAN-PCI interface, channel 3
            /// </summary>
            const PCANTP_PCIBUS3: TPCANTPHandle = $43;
            /// <summary>
            /// PCAN-PCI interface, channel 4
            /// </summary>
            const PCANTP_PCIBUS4: TPCANTPHandle = $44;
            /// <summary>
            /// PCAN-PCI interface, channel 5
            /// </summary>
            const PCANTP_PCIBUS5: TPCANTPHandle = $45;
            /// <summary>
            /// PCAN-PCI interface, channel 6
            /// </summary>
            const PCANTP_PCIBUS6: TPCANTPHandle = $46;
            /// <summary>
            /// PCAN-PCI interface, channel 7
            /// </summary>
            const PCANTP_PCIBUS7: TPCANTPHandle = $47;
            /// <summary>
            /// PCAN-PCI interface, channel 8
            /// </summary>
            const PCANTP_PCIBUS8: TPCANTPHandle = $48;
            /// <summary>
            /// PCAN-PCI interface, channel 9
            /// </summary>
            const PCANTP_PCIBUS9: TPCANTPHandle = $409;
            /// <summary>
            /// PCAN-PCI interface, channel 10
            /// </summary>
            const PCANTP_PCIBUS10: TPCANTPHandle = $40A;
            /// <summary>
            /// PCAN-PCI interface, channel 11
            /// </summary>
            const PCANTP_PCIBUS11: TPCANTPHandle = $40B;
            /// <summary>
            /// PCAN-PCI interface, channel 12
            /// </summary>
            const PCANTP_PCIBUS12: TPCANTPHandle = $40C;
            /// <summary>
            /// PCAN-PCI interface, channel 13
            /// </summary>
            const PCANTP_PCIBUS13: TPCANTPHandle = $40D;
            /// <summary>
            /// PCAN-PCI interface, channel 14
            /// </summary>
            const PCANTP_PCIBUS14: TPCANTPHandle = $40E;
            /// <summary>
            /// PCAN-PCI interface, channel 15
            /// </summary>
            const PCANTP_PCIBUS15: TPCANTPHandle = $40F;
            /// <summary>
            /// PCAN-PCI interface, channel 16
            /// </summary>
            const PCANTP_PCIBUS16: TPCANTPHandle = $410;

            /// <summary>
            /// PCAN-USB interface, channel 1
            /// </summary>
            const PCANTP_USBBUS1: TPCANTPHandle = $51;
            /// <summary>
            /// PCAN-USB interface, channel 2
            /// </summary>
            const PCANTP_USBBUS2: TPCANTPHandle = $52;
            /// <summary>
            /// PCAN-USB interface, channel 3
            /// </summary>
            const PCANTP_USBBUS3: TPCANTPHandle = $53;
            /// <summary>
            /// PCAN-USB interface, channel 4
            /// </summary>
            const PCANTP_USBBUS4: TPCANTPHandle = $54;
            /// <summary>
            /// PCAN-USB interface, channel 5
            /// </summary>
            const PCANTP_USBBUS5: TPCANTPHandle = $55;
            /// <summary>
            /// PCAN-USB interface, channel 6
            /// </summary>
            const PCANTP_USBBUS6: TPCANTPHandle = $56;
            /// <summary>
            /// PCAN-USB interface, channel 7
            /// </summary>
            const PCANTP_USBBUS7: TPCANTPHandle = $57;
            /// <summary>
            /// PCAN-USB interface, channel 8
            /// </summary>
            const PCANTP_USBBUS8: TPCANTPHandle = $58;
            /// <summary>
            /// PCAN-USB interface, channel 9
            /// </summary>
            const PCANTP_USBBUS9: TPCANTPHandle = $509;
            /// <summary>
            /// PCAN-USB interface, channel 10
            /// </summary>
            const PCANTP_USBBUS10: TPCANTPHandle = $50A;
            /// <summary>
            /// PCAN-USB interface, channel 11
            /// </summary>
            const PCANTP_USBBUS11: TPCANTPHandle = $50B;
            /// <summary>
            /// PCAN-USB interface, channel 12
            /// </summary>
            const PCANTP_USBBUS12: TPCANTPHandle = $50C;
            /// <summary>
            /// PCAN-USB interface, channel 13
            /// </summary>
            const PCANTP_USBBUS13: TPCANTPHandle = $50D;
            /// <summary>
            /// PCAN-USB interface, channel 14
            /// </summary>
            const PCANTP_USBBUS14: TPCANTPHandle = $50E;
            /// <summary>
            /// PCAN-USB interface, channel 15
            /// </summary>
            const PCANTP_USBBUS15: TPCANTPHandle = $50F;
            /// <summary>
            /// PCAN-USB interface, channel 16
            /// </summary>
            const PCANTP_USBBUS16: TPCANTPHandle = $510;

            /// <summary>
            /// PCAN-PC Card interface, channel 1
            /// </summary>
            const PCANTP_PCCBUS1: TPCANTPHandle = $61;
            /// <summary>
            /// PCAN-PC Card interface, channel 2
            /// </summary>
            const PCANTP_PCCBUS2: TPCANTPHandle = $62;

            /// <summary>
            /// PCAN-LAN interface, channel 1
            /// </summary>
            const PCANTP_LANBUS1: TPCANTPHandle = $801;
            /// <summary>
            /// PCAN-LAN interface, channel 2
            /// </summary>
            const PCANTP_LANBUS2: TPCANTPHandle = $802;
            /// <summary>
            /// PCAN-LAN interface, channel 3
            /// </summary>
            const PCANTP_LANBUS3: TPCANTPHandle = $803;
            /// <summary>
            /// PCAN-LAN interface, channel 4
            /// </summary>
            const PCANTP_LANBUS4: TPCANTPHandle = $804;
            /// <summary>
            /// PCAN-LAN interface, channel 5
            /// </summary>
            const PCANTP_LANBUS5: TPCANTPHandle = $805;
            /// <summary>
            /// PCAN-LAN interface, channel 6
            /// </summary>
            const PCANTP_LANBUS6: TPCANTPHandle = $806;
            /// <summary>
            /// PCAN-LAN interface, channel 7
            /// </summary>
            const PCANTP_LANBUS7: TPCANTPHandle = $807;
            /// <summary>
            /// PCAN-LAN interface, channel 8
            /// </summary>
            const PCANTP_LANBUS8: TPCANTPHandle = $808;
            /// <summary>
            /// PCAN-LAN interface, channel 9
            /// </summary>
            const PCANTP_LANBUS9: TPCANTPHandle = $809;
            /// <summary>
            /// PCAN-LAN interface, channel 10
            /// </summary>
            const PCANTP_LANBUS10: TPCANTPHandle = $80A;
            /// <summary>
            /// PCAN-LAN interface, channel 11
            /// </summary>
            const PCANTP_LANBUS11: TPCANTPHandle = $80B;
            /// <summary>
            /// PCAN-LAN interface, channel 12
            /// </summary>
            const PCANTP_LANBUS12: TPCANTPHandle = $80C;
            /// <summary>
            /// PCAN-LAN interface, channel 13
            /// </summary>
            const PCANTP_LANBUS13: TPCANTPHandle = $80D;
            /// <summary>
            /// PCAN-LAN interface, channel 14
            /// </summary>
            const PCANTP_LANBUS14: TPCANTPHandle = $80E;
            /// <summary>
            /// PCAN-LAN interface, channel 15
            /// </summary>
            const PCANTP_LANBUS15: TPCANTPHandle = $80F;
            /// <summary>
            /// PCAN-LAN interface, channel 16
            /// </summary>
            const PCANTP_LANBUS16: TPCANTPHandle = $810;


            /// <summary>
            /// Clock frequency in Herz (80000000, 60000000, 40000000, 30000000, 24000000, 20000000)
            /// </summary>
            const PCANTP_BR_CLOCK: String = 'f_clock';
            /// <summary>
            /// Clock frequency in Megaherz (80, 60, 40, 30, 24, 20)
            /// </summary>
            const PCANTP_BR_CLOCK_MHZ: String = 'f_clock_mhz';
            /// <summary>
            /// Clock prescaler for nominal time quantum
            /// </summary>
            const PCANTP_BR_NOM_BRP: String = 'nom_brp';
            /// <summary>
            /// TSEG1 segment for nominal bit rate in time quanta
            /// </summary>
            const PCANTP_BR_NOM_TSEG1: String = 'nom_tseg1';
            /// <summary>
            /// TSEG2 segment for nominal bit rate in time quanta
            /// </summary>
            const PCANTP_BR_NOM_TSEG2: String = 'nom_tseg2';
            /// <summary>
            /// Synchronization Jump Width for nominal bit rate in time quanta
            /// </summary>
            const PCANTP_BR_NOM_SJW: String = 'nom_sjw';
            /// <summary>
            /// Sample point for nominal bit rate
            /// </summary>
            const PCANTP_BR_NOM_SAMPLE: String = 'nom_sam';
            /// <summary>
            /// Clock prescaler for highspeed data time quantum
            /// </summary>
            const PCANTP_BR_DATA_BRP: String = 'data_brp';
            /// <summary>
            /// TSEG1 segment for fast data bit rate in time quanta
            /// </summary>
            const PCANTP_BR_DATA_TSEG1: String = 'data_tseg1';
            /// <summary>
            /// TSEG2 segment for fast data bit rate in time quanta
            /// </summary>
            const PCANTP_BR_DATA_TSEG2: String = 'data_tseg2';
            /// <summary>
            /// Synchronization Jump Width for highspeed data bit rate in time quanta
            /// </summary>
            const PCANTP_BR_DATA_SJW: String = 'data_sjw';
            /// <summary>
            /// Secondary sample point delay for highspeed data bit rate in cyles
            /// </summary>
            const PCANTP_BR_DATA_SAMPLE: String = 'data_ssp_offset';


            /// <summary>
            /// No debug messages
            /// </summary>
            const PCANTP_DEBUG_NONE: Byte = 0;
            /// <summary>
            /// Puts CAN debug messages to stdout
            /// </summary>
            const PCANTP_DEBUG_CAN: Byte = 1;
            /// <summary>
            /// The Channel is illegal or not available
            /// </summary>
            const PCANTP_CHANNEL_UNAVAILABLE: Byte = 0;
            /// <summary>
            /// The Channel is available
            /// </summary>
            const PCANTP_CHANNEL_AVAILABLE: Byte = 1;
            /// <summary>
            /// The Channel is valid, and is being used
            /// </summary>
            const PCANTP_CHANNEL_OCCUPIED: Byte = 2;

            /// <summary>
            /// if set Flow Control frame shall not use the WT flow status value
            /// </summary>
            const PCANTP_WFT_MAX_UNLIMITED: Byte = $00;

            /// <summary>
            /// An integer describing the Wait Frame Transmissions parameter.
            /// </summary>
            const PCANTP_WFT_MAX_DEFAULT: Byte = $10;

            /// <summary>
            /// Hide messages with type PCANTP_MESSAGE_INDICATION from CANTP_Read function
            /// </summary>
            const PCANTP_MSG_PENDING_HIDE: Byte = $00;
            /// <summary>
            /// Show messages with type PCANTP_MESSAGE_INDICATION from CANTP_Read function
            /// </summary>
            const PCANTP_MSG_PENDING_SHOW: Byte = $01;

            /// <summary>
            /// Uses CAN frame data optimization
            /// </summary>
            const PCANTP_CAN_DATA_PADDING_NONE: Byte = $00;
            /// <summary>
            /// Uses CAN frame data padding (default, i.e. CAN DLC = 8)
            /// </summary>
            const PCANTP_CAN_DATA_PADDING_ON: Byte = $01;
            /// <summary>
            /// Default value used if CAN data padding is enabled
            /// </summary>
            const PCANTP_CAN_DATA_PADDING_VALUE: Byte = $55;

            /// <summary>
            /// Disable reception of unformatted (NON-ISO-TP) CAN frames (default):
            /// only ISO 15765 messages will be received
            /// </summary>
            const PCANTP_CAN_UNSEGMENTED_OFF: Byte = $00;
            /// <summary>
            /// Enable reception of unformatted (NON-ISO-TP) CAN frames:
            /// received messages will be treated as either ISO 15765 or as an unformatted CAN frame
            /// </summary>
            const PCANTP_CAN_UNSEGMENTED_ON: Byte = $01;
            /// <summary>
            /// Enable reception of unformatted (NON-ISO-TP) CAN frames:
            /// received messages will be treated as ISO 15765, unformatted CAN frame, or both (user will able to read fragmented CAN frames
            /// </summary>
            const PCANTP_CAN_UNSEGMENTED_ALL_FRAMES: Byte = $02;

            /// <summary>
            /// Default priority for ISO-TP messages (only available for normal fixed, mixed and enhanced addressing)
            /// </summary>
            const PCANTP_J1939_PRIORITY_DEFAULT: Byte = $06;

            /// <summary>
            /// Mask to retrieve the CAN ID type
            /// </summary>
            const PCANTP_ID_CAN_MASK: Byte = $0F;
            /// <summary>
            /// Mask to check if the priority field is set
            /// </summary>
            const PCANTP_ID_CAN_IS_PRIORITY_MASK: Byte = $10;
            /// <summary>
            /// Mask to retrieve the J1939 priority
            /// </summary>
            const PCANTP_ID_CAN_PRIORITY_MASK: Byte = $E0;

            /// <summary>
            /// Constant defining that no CAN ID mapping exists
            /// </summary>
            const CAN_ID_NO_MAPPING: LongWord = LongWord($FFFFFFFF);

            /// <summary>
            /// max data length prior ISO 15765-2:2016 update
            /// </summary>
            const PCANTP_MESSAGE_MAX_LENGTH: LongWord = LongWord($FFF);


        /// <summary>
        /// Initializes a PCANTP-Client based on a CANTP Channel
        /// </summary>
        /// <remarks>Only one PCANTP-Client can be initialized per CAN-Channel</remarks>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <param name="Baudrate">The CAN Hardware speed</param>
        /// <param name="HwType">NON PLUG&PLAY: The type of hardware and operation mode</param>
        /// <param name="IOPort">NON PLUG&PLAY: The I/O address for the parallel port</param>
        /// <param name="Interrupt">NON PLUG&PLAY: Interrupt number of the parallel port</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function Initialize(
            CanChannel: TPCANTPHandle;
            Baudrate: TPCANTPBaudrate;
            HwType: TPCANTPHWType;
            IOPort: LongWord;
            Interrupt: Word
            ): TPCANTPStatus; overload;

        /// <summary>
        /// Initializes a PCANTP-Client based on a CANTP Channel
        /// </summary>
        /// <remarks>Only one PCANTP-Client can be initialized per CAN-Channel</remarks>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function Initialize(
            CanChannel: TPCANTPHandle;
            Baudrate: TPCANTPBaudrate
            ): TPCANTPStatus; overload;

        /// <summary>
        /// Initializes a PCANTP-Client based on a CANTP Channel
        /// </summary>
        /// <remarks>Only one PCANTP-Client can be initialized per CAN-Channel</remarks>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <param name="Baudrate">The CAN Hardware speed</param>
        /// <param name="HwType">NON PLUG&PLAY: The type of hardware and operation mode</param>
        /// <param name="IOPort">NON PLUG&PLAY: The I/O address for the parallel port</param>
        /// <param name="Interrupt">NON PLUG&PLAY: Interrupt number of the parallel port</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function InitializeFD(
            CanChannel: TPCANTPHandle;
            BitrateFD: TPCANTPBitrateFD
            ): TPCANTPStatus;

        /// <summary>
        /// Uninitializes a PCANTP-Client initialized before
        /// </summary>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function Uninitialize(
            CanChannel: TPCANTPHandle
            ): TPCANTPStatus;


        /// <summary>
        /// Resets the receive and transmit queues of a PCANTP-Client
        /// </summary>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function Reset(
            CanChannel: TPCANTPHandle
            ): TPCANTPStatus;

        /// <summary>
        /// Gets information about the internal BUS status of a PCANTP-Channel.
        /// </summary>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function GetStatus(
            CanChannel: TPCANTPHandle
            ): TPCANTPStatus;

        /// <summary>
        /// Reads a PCANTP message from the receive queue of a PCANTP-Client
        /// </summary>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <param name="MessageBuffer">A TPCANTPMsg structure buffer to store the PUDS message</param>
        /// <param name="TimestampBuffer">"A TPCANTPTimestamp structure buffer to get
        /// the reception time of the message. If this value is not desired, this parameter
        /// should be passed as NULL"</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function Read(
            CanChannel: TPCANTPHandle;
            var MessageBuffer: TPCANTPMsg;
                  TimestampBuffer: PTPCANTPTimestamp
            ): TPCANTPStatus; overload;
        /// <summary>
        /// Reads a PCANTP message from the receive queue of a PCANTP-Client
        /// </summary>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <param name="MessageBuffer">A TPCANTPMsg structure buffer to store the PUDS message</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function Read(
            CanChannel: TPCANTPHandle;
            var MessageBuffer: TPCANTPMsg
            ): TPCANTPStatus; overload;

        /// <summary>
        /// Transmits a PCANTP message
        /// </summary>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <param name="MessageBuffer">A TPCANTPMsg buffer with the message to be sent</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function Write(
            CanChannel: TPCANTPHandle;
            var MessageBuffer: TPCANTPMsg
            ): TPCANTPStatus;

        /// <summary>
        /// Retrieves a PCANTP-Client value
        /// </summary>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <param name="Parameter">The TPCANTPParameter parameter to get</param>
        /// <param name="NumericBuffer">Buffer for the parameter value</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function GetValue(
            CanChannel: TPCANTPHandle;
            Parameter: TPCANTPParameter;
            NumericBuffer: PLongWord;
            BufferLength: LongWord
            ): TPCANTPStatus; overload;
        /// <summary>
        /// Retrieves a PCANTP-Client value
        /// </summary>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <param name="Parameter">The TPCANTPParameter parameter to get</param>
        /// <param name="StringBuffer">Buffer for the parameter value</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function GetValue(
            CanChannel: TPCANTPHandle;
            Parameter: TPCANTPParameter;
            StringBuffer: PAnsiChar;
            BufferLength: LongWord
            ): TPCANTPStatus; overload;

        /// <summary>
        /// Configures or sets a PCANTP-Client value
        /// </summary>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <param name="Parameter">The TPCANTPParameter parameter to set</param>
        /// <param name="NumericBuffer">Buffer with the value to be set</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function SetValue(
            CanChannel: TPCANTPHandle;
            Parameter: TPCANTPParameter;
            NumericBuffer: PLongWord;
            BufferLength: LongWord
            ): TPCANTPStatus; overload;
        /// <summary>
        /// Configures or sets a PCANTP-Client value
        /// </summary>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <param name="Parameter">The TPCANTPParameter parameter to set</param>
        /// <param name="StringBuffer">Buffer with the value to be set</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function SetValue(
            CanChannel: TPCANTPHandle;
            Parameter: TPCANTPParameter;
            StringBuffer: PAnsiChar;
            BufferLength: LongWord
            ): TPCANTPStatus; overload;

        /// <summary>
        /// Adds a user-defined PCAN-TP mapping between CAN ID and Network Address Information
        /// </summary>
        /// <remark>
        /// Defining a mapping enables ISO-TP communication with 11BITS CAN ID or
        /// with opened Addressing Formats (like PCANTP_FORMAT_NORMAL or PCANTP_FORMAT_EXTENDED).
        /// </remark>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <param name="canID">The CAN ID to map</param>
        /// <param name="canIDResponse">The CAN ID mapped response</param>
        /// <param name="canIdType">CAN ID Type (11 or 29 bits, see PCANTP_ID_CAN_XXX)</param>
        /// <param name="formatType">Addressing format (see PCANTP_FORMAT_XXX)</param>
        /// <param name="msgType">Message type (remote or diagnostic, see PCANTP_MESSAGE_XXX)</param>
        /// <param name="sourceAddr">Source Address</param>
        /// <param name="targetAddr">Target Address</param>
        /// <param name="targetType">Target Addressing Type (physical or functional, see PCANTP_ADDRESSING_XXX)</param>
        /// <param name="remoteAddr">Address extension or Remote address</param>
        /// <returns>A TPCANTPStatus code : PCANTP_ERROR_OK is returned on success,
        /// PCANTP_ERROR_WRONG_PARAM states invalid Network Address Information parameters.</returns>
        class function AddMapping(
            CanChannel: TPCANTPHandle;
            canID: LongWord;
            canIDResponse: LongWord;
            canIdType: TPCANTPIdType;
            formatType: TPCANTPFormatType;
            msgType: TPCANTPMessageType ;
            sourceAddr: Byte;
            targetAddr: Byte;
            targetType: TPCANTPAddressingType;
            remoteAddr: Byte
            ): TPCANTPStatus;

        /// <summary>
        /// Removes a user-defined PCAN-TP mapping between CAN ID and Network Address Information
        /// </summary>
        /// <param name="CanChannel">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <param name="canID">The mapped CAN ID to remove</param>
        /// <returns>A TPCANTPStatus code. PCANTP_ERROR_OK is returned on success</returns>
        class function RemoveMapping(
            CanChannel: TPCANTPHandle;
            canID: LongWord
            ): TPCANTPStatus;

        /// <summary>
        /// Returns a descriptive text of a given TPCANTPStatus error
        /// code, in any desired language
        /// </summary>
        /// <remarks>The current languages available for translation are:
        /// Neutral (0x00), German (0x07), English (0x09), Spanish (0x0A),
        /// Italian (0x10) and French (0x0C)</remarks>
        /// <param name="Error">A TPCANTPStatus error code</param>
        /// <param name="Language">Indicates a 'Primary language ID'</param>
        /// <param name="StringBuffer">Buffer for the text (must be at least 256 in length)</param>
        /// <returns>A TPCANTPStatus error code</returns>
        class function GetErrorText(
            Error: TPCANTPStatus;
            Language: Word;
            StringBuffer: PAnsiChar
            ): TPCANTPStatus;

        // Get the 29bits CAN ID type with a specific J1939 priority
        class function PCANTP_ID_CAN_GET_29B(j1939_priority: Byte): TPCANTPIdType;
        // Retrieves the priority field from a CAN ID type
        class function PCANTP_ID_CAN_GET_PRIORIY(id_type: Byte): Byte;
        // States if the CAN ID TYpe is 29bits
        class function PCANTP_ID_CAN_IS_EXTENDED(id_type: Byte): Boolean;
        // States if the id_type contains a J1939 priority field
        class function PCANTP_ID_CAN_HAS_PRIORITY(id_type: Byte): Boolean;

        end;

implementation
uses SysUtils;
const DLL_Name = 'PCAN-ISO-TP.DLL';

function CANTP_Initialize(Channel: TPCANTPHandle; Btr0Btr1: TPCANTPBaudrate; HwType: TPCANTPHWType; IOPort: LongWord; Interrupt: Word): TPCANTPStatus; stdcall;
external DLL_Name;
function CANTP_InitializeFD(Channel: TPCANTPHandle; BitrateFD: TPCANTPBitrateFD): TPCANTPStatus; stdcall;
external DLL_Name;
function CANTP_Uninitialize(Channel: TPCANTPHandle): TPCANTPStatus; stdcall;
external DLL_Name;
function CANTP_Reset(Channel: TPCANTPHandle): TPCANTPStatus; stdcall;
external DLL_Name;
function CANTP_GetStatus(Channel: TPCANTPHandle): TPCANTPStatus; stdcall;
external DLL_Name;
function CANTP_Read(Channel: TPCANTPHandle; var MessageBuffer: TPCANTPMsg; TimestampBuffer: PTPCANTPTimestamp):TPCANTPStatus; stdcall;
external DLL_Name;
function CANTP_Write(Channel: TPCANTPHandle; var MessageBuffer: TPCANTPMsg): TPCANTPStatus; stdcall;
external DLL_Name;
function CANTP_GetValue(Channel: TPCANTPHandle; Parameter: TPCANTPParameter; Buffer: Pointer; BufferLength: LongWord): TPCANTPStatus; stdcall;
external DLL_Name;
function CANTP_SetValue(Channel: TPCANTPHandle; Parameter: TPCANTPParameter; Buffer: Pointer; BufferLength: LongWord): TPCANTPStatus; stdcall;
external DLL_Name;
function CANTP_AddMapping(Channel: TPCANTPHandle; canID: LongWord; canIDResponse: LongWord; canIdType: TPCANTPIdType; formatType: TPCANTPFormatType; msgType: TPCANTPMessageType ; sourceAddr: Byte; targetAddr: Byte; targetType: TPCANTPAddressingType; remoteAddr: Byte): TPCANTPStatus; stdcall;
external DLL_Name;
function CANTP_RemoveMapping(Channel: TPCANTPHandle; canID: LongWord): TPCANTPStatus; stdcall;
external DLL_Name;
function CANTP_GetErrorText(Error: TPCANTPStatus; Language: Word; StringBuffer: PAnsiChar): TPCANTPStatus; stdcall;
external DLL_Name;

class function TCanTpApi.Initialize(CanChannel: TPCANTPHandle; Baudrate: TPCANTPBaudrate; HwType: TPCANTPHWType; IOPort: LongWord; Interrupt: Word): TPCANTPStatus;
begin
    Result:= CANTP_Initialize(CanChannel, Baudrate, HwType, IOPort, Interrupt);
end;

class function TCanTpApi.Initialize(CanChannel: TPCANTPHandle; Baudrate: TPCANTPBaudrate): TPCANTPStatus;
begin
    Result:= CANTP_Initialize(CanChannel, Baudrate, TPCANTPHWType(0), 0, 0);
end;

class function TCanTpApi.InitializeFD(CanChannel: TPCANTPHandle; BitrateFD: TPCANTPBitrateFD): TPCANTPStatus;
begin
    Result:= CANTP_InitializeFD(CanChannel, BitrateFD);
end;

class function TCanTpApi.Uninitialize(CanChannel: TPCANTPHandle): TPCANTPStatus;
begin
    Result:= CANTP_Uninitialize(CanChannel);
end;

class function TCanTpApi.Reset(CanChannel: TPCANTPHandle): TPCANTPStatus;
begin
    Result:= CANTP_Reset(CanChannel);
end;

class function TCanTpApi.GetStatus(CanChannel: TPCANTPHandle): TPCANTPStatus;
begin
    Result:= CANTP_GetStatus(CanChannel);
end;

class function TCanTpApi.Read(CanChannel: TPCANTPHandle; var MessageBuffer: TPCANTPMsg; TimestampBuffer: PTPCANTPTimestamp): TPCANTPStatus;
begin
    Result:= CANTP_Read(CanChannel, MessageBuffer, TimestampBuffer);
end;
class function TCanTpApi.Read(CanChannel: TPCANTPHandle; var MessageBuffer: TPCANTPMsg): TPCANTPStatus;
begin
    Result:= CANTP_Read(CanChannel, MessageBuffer, nil);
end;

class function TCanTpApi.Write(CanChannel: TPCANTPHandle; var MessageBuffer: TPCANTPMsg): TPCANTPStatus;
begin
    Result:= CANTP_Write(CanChannel, MessageBuffer);
end;

class function TCanTpApi.GetValue(CanChannel: TPCANTPHandle; Parameter: TPCANTPParameter; NumericBuffer: PLongWord; BufferLength: LongWord): TPCANTPStatus;
begin
    Result:= CANTP_GetValue(CanChannel, Parameter, NumericBuffer, BufferLength);
end;

class function TCanTpApi.GetValue(CanChannel: TPCANTPHandle; Parameter: TPCANTPParameter; StringBuffer: PAnsiChar; BufferLength: LongWord): TPCANTPStatus;
begin
    Result:= CANTP_GetValue(CanChannel, Parameter, StringBuffer, BufferLength);
end;

class function TCanTpApi.SetValue(CanChannel: TPCANTPHandle; Parameter: TPCANTPParameter; NumericBuffer: PLongWord; BufferLength: LongWord): TPCANTPStatus;
begin
    Result:= CANTP_SetValue(CanChannel, Parameter, NumericBuffer, BufferLength);
end;

class function TCanTpApi.SetValue(CanChannel: TPCANTPHandle; Parameter: TPCANTPParameter; StringBuffer: PAnsiChar; BufferLength: LongWord): TPCANTPStatus;
begin
    Result:= CANTP_SetValue(CanChannel, Parameter, StringBuffer, BufferLength);
end;

class function TCanTpApi.AddMapping(CanChannel: TPCANTPHandle; canID: LongWord; canIDResponse: LongWord; canIdType: TPCANTPIdType; formatType: TPCANTPFormatType; msgType: TPCANTPMessageType ; sourceAddr: Byte; targetAddr: Byte; targetType: TPCANTPAddressingType; remoteAddr: Byte): TPCANTPStatus;
begin
    Result:= CANTP_AddMapping(CanChannel, canID, canIDResponse, canIdType, formatType, msgType, sourceAddr, targetAddr, targetType, remoteAddr);
end;

class function TCanTpApi.RemoveMapping(CanChannel: TPCANTPHandle; canID: LongWord): TPCANTPStatus;
begin
    Result:= CANTP_RemoveMapping(CanChannel, canID);
end;

class function TCanTpApi.GetErrorText(Error: TPCANTPStatus; Language: Word; StringBuffer: PAnsiChar): TPCANTPStatus;
begin
    Result:= CANTP_GetErrorText(Error, Language, StringBuffer);
end;

class function TCanTpApi.PCANTP_ID_CAN_GET_29B(j1939_priority: Byte): TPCANTPIdType;
begin
  Result := TPCANTPIdType((j1939_priority shl 5) or TCanTpApi.PCANTP_ID_CAN_IS_PRIORITY_MASK or (Byte(PCANTP_ID_CAN_29BIT) and TCanTpApi.PCANTP_ID_CAN_MASK));
end;

class function TCanTpApi.PCANTP_ID_CAN_GET_PRIORIY(id_type: Byte): Byte;
begin
  Result := Byte(id_type shr 5);
end;

class function TCanTpApi.PCANTP_ID_CAN_IS_EXTENDED(id_type: Byte): Boolean;
begin
  Result := ((id_type and Byte(PCANTP_ID_CAN_29BIT)) = Byte(PCANTP_ID_CAN_29BIT));
end;

class function TCanTpApi.PCANTP_ID_CAN_HAS_PRIORITY(id_type: Byte): Boolean;
begin
  Result := (PCANTP_ID_CAN_IS_EXTENDED(id_type) and ((id_type and TCanTpApi.PCANTP_ID_CAN_IS_PRIORITY_MASK) = TCanTpApi.PCANTP_ID_CAN_IS_PRIORITY_MASK));
end;

end.
